/**********************************************************************************************************************
*
* Copyright (C) 2012 Continental Automotive Systems, Inc.
*
* Author: Jean-Pierre.Bogler@continental-corporation.com
*
* Header for the NodestateMachine stub.
*
* The header file defines the interfaces offered by the NodeStateMachine stub.
*
* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/.
*
* Date             Author              Reason
* 27.09.2012       Jean-Pierre Bogler  CSP_WZ#1194: Initial creation.
* 24.10.2012       Jean-Pierre Bogler  CSP_WZ#1322: Changed parameter types of interface functions.
*
**********************************************************************************************************************/


/**********************************************************************************************************************
*
* Header includes
*
**********************************************************************************************************************/

#include "NodeStateMachine.h" /* own header file            */
#include "NodeStateManager.h"
#include "NodeStateTypes.h"
#include <stdio.h>
#include <dlt/dlt.h>


static const char* g_pcszNsmRestartReason[] =
{
    "NotSet",                         /**< Initial value when reset reason is not set           */
    "ApplicationFailure",             /**< Reset was requested by System Health Mon.            */
	"NsmRestartReason_CPUOverLimit",  /**< Reset was requested by System Health Mon.            */
	"NsmRestartReason_MEM_OverLimit", /**< Reset was requested by System Health Mon.            */
    "Diagnosis",                      /**< Reset was requested by diagnosis                     */
    "SWL application",                /**< Reset was requested by the SWL application           */
    "User application",               /**< Reset was requested by an user application           */
    "Last valid reset reasons"        /**< Last value to identify valid reset reasons           */
};

/**********************************************************************************************************************
*
* Local defines, macros, constants and type definitions.
*
**********************************************************************************************************************/

/* There are currently no local defines, macros or types */


/**********************************************************************************************************************
*
* Local variables
*
**********************************************************************************************************************/

/* Import NSM DLT context */
DLT_IMPORT_CONTEXT(NsmContext)

/**********************************************************************************************************************
*
* Prototypes for file local functions (see implementation for description)
*
**********************************************************************************************************************/

/* There are no file local functions */

/**********************************************************************************************************************
*
* Local (static) functions
*
**********************************************************************************************************************/

/* There are no file local functions */

/**********************************************************************************************************************
*
* Interfaces, exported functions. See header for detailed description.
*
**********************************************************************************************************************/

unsigned char NsmcInit(void)
{
  printf("NSMC: NsmcInit called.\n");

  return 1;
}


unsigned char NsmcLucRequired(void)
{
  printf("NSMC: NsmcLucRequired called.\n");

  return 1;
}


NsmErrorStatus_e NsmcSetData(NsmDataType_e enData, unsigned char *pData, unsigned int u32DataLen)
{
  /* Function local variables                                        */
  NsmErrorStatus_e enRetVal = NsmErrorStatus_NotSet; /* Return value */

  /* Check which data the NSMC wants to set */
  switch(enData)
  {
	/* NSMC wants to set the NodeState */
	case NsmDataType_NodeState:
	  enRetVal = NsmcSetNodeState(enData, pData, u32DataLen);
	break;

	case NsmDataType_AppMode:
	case NsmDataType_RestartReason:
	case NsmDataType_SessionState:
	case NsmDataType_ShutdownReason:
	case NsmDataType_BootMode:
	case NsmDataType_RunningReason:
	case NsmDataType_RegisterSession:
	case NsmDataType_UnRegisterSession:
	case NsmDataType_DimmingLevel:
	case NsmDataType_AudioProperty:
	default:
	  enRetVal = NsmErrorStatus_Parameter;
	break;
  }

  return enRetVal;
  return NsmErrorStatus_Ok;
}


unsigned char NsmcRequestNodeRestart(NsmRestartReason_e enRestartReason, unsigned int u32RestartType)
{
    DLT_LOG(NsmContext, DLT_LOG_INFO, DLT_STRING("NSMC: NsmcRequestNodeRestart called."); DLT_STRING(g_pcszNsmRestartReason[enRestartReason]); DLT_INT(u32RestartType));

    return 1;
}


unsigned int NsmcGetInterfaceVersion(void)
{
  DLT_LOG(NsmContext, DLT_LOG_INFO, DLT_STRING("NSMC: NsmcGetInterfaceVersion called."));

  return NSMC_INTERFACE_VERSION;
}

NsmErrorStatus_e NsmcSetNodeState(NsmDataType_e enData, unsigned char *pData, unsigned int u32DataLen)
{
  NsmErrorStatus_e enRetVal = NsmErrorStatus_NotSet; /* Return value */
  /* All things will be protected under NSM__pNodeStateMutex in NSM__enSetNodeState function */

  if (NULL == pData)
  {
    DLT_LOG(NsmContext, DLT_LOG_ERROR, DLT_STRING("NSMC: NULL pointer at "),
                                       DLT_STRING(__FILE__),
                                       DLT_STRING(": "),
                                       DLT_STRING(__func__),
                                       DLT_STRING(": "),
                                       DLT_INT((gint)__LINE__),
                                       DLT_STRING("."));
    enRetVal = NsmErrorStatus_Error;
  }

  NsmNodeState_e enNodeState = *((NsmNodeState_e *)pData);
  switch(enNodeState)
  {
	case NsmNodeState_Shutdown:
		(void)NsmcExecProcessSync("/bin/sync");
		enRetVal = NsmErrorStatus_Ok;
	break;

	case NsmNodeState_StartUp:
	case NsmNodeState_BaseRunning:
	case NsmNodeState_LucRunning:
	case NsmNodeState_FullyRunning:
	case NsmNodeState_FullyOperational:
	case NsmNodeState_ShuttingDown:
	case NsmNodeState_ShutdownDelay:
	case NsmNodeState_FastShutdown:
	case NsmNodeState_DegradedPower:
	case NsmNodeState_Last:
    case NsmNodeState_NotSet:
		enRetVal = NsmErrorStatus_Ok;
	break;

	default:
	  enRetVal = NsmErrorStatus_Parameter;
	break;

  }

  return enRetVal;
}


gboolean NsmcExecProcessSync(gchar *process)
{
  gchar    *argv[]      = {process, NULL};
  GError   *error       = NULL;
  gint      proc_retval = 0;
  gboolean  proc_ok     = FALSE;

  /* Start the process.*/
  (void) g_spawn_sync(NULL,
                     argv,
                     NULL,
                     G_SPAWN_STDOUT_TO_DEV_NULL|G_SPAWN_STDERR_TO_DEV_NULL,
                     NULL,
                     NULL,
                     NULL,
                     NULL,
                     &proc_retval,
                     &error);

  /* Check for errors of prg execution. */
  if(error == NULL)
  {
	DLT_LOG(NsmContext,
			DLT_LOG_INFO,
            DLT_STRING("NSMC: NsmcExecProcessSync: ");
            DLT_STRING(process);
            DLT_STRING(" -> success."));
    /* Prog. executed. Check that return value is '0' */
    proc_ok = (proc_retval == 0);
  }
  else
  {
    /* Prog. execution failed. */
    proc_ok = FALSE;
    DLT_LOG(NsmContext,
            DLT_LOG_ERROR,
            DLT_STRING("NSMC: NsmcExecProcessSync: ");
            DLT_STRING(process);
            DLT_STRING(" -> failed");
            DLT_STRING("Reason:"); DLT_STRING(error->message));
    g_error_free(error);
  }

  return proc_ok;
}

