#ifndef NODESTATETYPES_H
#define NODESTATETYPES_H

/**********************************************************************************************************************
*
* Copyright (C) 2012 Continental Automotive Systems, Inc.
*
* Author: Jean-Pierre.Bogler@continental-corporation.com
*
* Type and constant definitions to communicate with the NSM.
*
* The file defines types and constants to be able to communicate with the NSM.
*
* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/.
*
* Date       Author             Reason
* 2012.09.27 uidu5846  1.0.0.0  CSP_WZ#1194: Introduced 'NodeStateTypes.h' to avoid circle includes
*                                            and encapsulate type definitions.
* 2012.10.24 uidu5846  1.0.0.1  CSP_WZ#1322: Removed "ssw_types" redefinition from header.
*                                            Since the same native types are used, no interface change.
* 2013.04.18 uidu5846  1.2.1    OvipRbt#1153 Added possibility to register sessions via the NSMC.
*
**********************************************************************************************************************/

/** \ingroup SSW_LCS */
/** \defgroup SSW_NSM_TEMPLATE Node State Manager
 *  \{
 */
/** \defgroup SSW_NSM_INTERFACE API document
 *  \{
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**********************************************************************************************************************
*
*  CONSTANTS
*
**********************************************************************************************************************/

/* Definition for bit value */
#define NSM_BIT_VALUE(x)          (unsigned int)(1<<x)

/* Definitions of DBus type */
#define LCM_BUS_TYPE              G_BUS_TYPE_SYSTEM

/* Definitions of D-Bus names and addresses */
#define NSM_BUS_TYPE              G_BUS_TYPE_SYSTEM                                /* Defines bus type according to GBusType  */
#define NSM_BUS_NAME              "org.genivi.NodeStateManager"                    /* The bus name of the NSM                 */
#define NSM_LC_CONTROL_OBJECT     "/org/genivi/NodeStateManager/LifecycleControl"  /* Address of the lifecycle Control object */
#define NSM_LC_CONSUMER_OBJECT    "/org/genivi/NodeStateManager/LifecycleConsumer" /* Address of the lifecycle Consumer object*/
#define NSM_CONSUMER_OBJECT       "/org/genivi/NodeStateManager/Consumer"          /* Address of the consumer object          */

/* Dbus type of Node Voltage Manager */
#define CVM_BUS_TYPE              LCM_BUS_TYPE
/* Dbus name of Critical Voltage Manager */
#define CVM_BUS_NAME              "com.bosch.CriticalVoltageManager"
/* Dbus object path of Critical Voltage Manager */
#define CVM_OBJECT_PATH           "/com/bosch/CriticalVoltageManager/Consumer"
/* Dbus interface name of Critical Voltage Manager */
#define CVM_INTERFACE_NAME        "com.bosch.CriticalVoltageManager.Consumer"

/* Dbus type of Heat Control Manager */
#define HCM_BUS_TYPE              LCM_BUS_TYPE
/* Dbus name of Heat Control Manager */
#define HCM_BUS_NAME              "com.bosch.HeatControlManager"
/* Dbus object path of Heat Control Manager */
#define HCM_OBJECT_PATH           "/com/bosch/HeatControlManager/Consumer"
/* Dbus interface name of Heat Control Manager */
#define HCM_INTERFACE_NAME        "com.bosch.HeatControlManager.Consumer"

/* DBus definition for DIMMING */
#define NSM_DIMMING_BUS_NAME      "com.bosch.dimming.dimming"
#define NSM_DIMMING_OBJECT        "/com/bosch/dimming"

/* DBus definition for AUDIO */
#define NSM_AUDIO_BUS_TYPE        LCM_BUS_TYPE
#define NSM_AUDIO_BUS_NAME        "org.genivi.audiomanager"
#define NSM_AUDIO_OBJECT          "/org/genivi/audiomanager/commandinterface"
#define NSM_AUDIO_INTERFACE       "org.genivi.audiomanager.commandinterface"

/* Defines for the unified interface caller */
#define NSM_CALLER_CVM                         (0)
#define NSM_CALLER_HCM                         (1)
#define NSM_CALLER_ACC                         (2)

/* Defines for the components to operate on */
#define NSM_DIMMING_COMPONENT                  (0)
#define NSM_AUDIO_COMPONENT                    (1)

/* Property definition for AUDIO */
#define NSM_AUDIO_SINK_ID                      (1)
#define NSM_AUDIO_SYSPROPERTY                  (1007)

/* Defines the ACC type trigger mode */
#define NSM_ACCTYPE_EDGE_TRIGGER_MODE          (0)
#define NSM_ACCTYPE_LEVEL_TRIGGER_MODE         (1)

/* Defines for session handling */
#define NSM_DEFAULT_SESSION_OWNER "NodeStateManager"           /**< "Owner" of the default sessions                  */

/* Defines for internal settings like max. string lenghts */
#define NSM_MAX_SESSION_NAME_LENGTH  256                       /**< Max. number of chars a session name can have     */
#define NSM_MAX_SESSION_OWNER_LENGTH 256                       /**< Max. number of chars for name of session owner   */

/*
 * Defines for shutdown handling as bit masks. Used to register for multiple shutdown types and as parameter to
 * inform clients about the shutdown type via the LifecycleConsumer interface.
 */
#define NSM_SHUTDOWNTYPE_NOT      0x00000000U                  /**< Client not registered for any shutdown           */
#define NSM_SHUTDOWNTYPE_NORMAL   0x00000001U                  /**< Client registered for normal shutdown            */
#define NSM_SHUTDOWNTYPE_FAST     0x00000002U                  /**< Client registered for fast shutdown              */
#define NSM_SHUTDOWNTYPE_RUNUP    0x80000000U                  /**< The shutdown type "run up" can not be used for
                                                                    registration. Clients which are registered and
                                                                    have been shut down, will automatically be
                                                                    informed about the "run up", when the shut down
                                                                    is canceled.                                    */

/*
Defines the cpu signal types for cpu monitor
It should be in accordance with the definition in NHM
*/
#define NSM_CPU_STATE_NORMAL            0
#define NSM_CPU_STATE_OVER_LIMIT        1
#define NSM_CPU_TYPE_SYSTEM             0
#define NSM_CPU_TYPE_PROC               1

/**********************************************************************************************************************
*
*  TYPE
*
**********************************************************************************************************************/

/**
 * The enumeration defines the different types of data that can be exchanged between the NodeStateManager (NSM)
 * and the NodeStateMachine (NSMC). Based on this value, the setter and getter functions of the NSM and NSMC will
 * interpret data behind the passed byte pointer.
 */
typedef enum _NsmDataType_e
{
  NsmDataType_AppMode,                 /**< An ApplicationMode should be set or get */
  NsmDataType_NodeState,               /**< A NodeState should be set or get        */
  NsmDataType_RestartReason,           /**< A RestartReason should be set or get    */
  NsmDataType_SessionState,            /**< A SessionState should be set or get     */
  NsmDataType_ShutdownReason,          /**< A ShutdownReason should be set or get   */
  NsmDataType_BootMode,                /**< A BootMode should be set or get         */
  NsmDataType_RunningReason,           /**< A RunningReason should be set or get    */
  NsmDataType_RegisterSession,         /**< A Session should be registered          */
  NsmDataType_UnRegisterSession,       /**< A Session should be unregistered        */
  NsmDataType_DimmingLevel,            /**< A DimmingLevel should be set or get     */
  NsmDataType_AudioProperty            /**< A AudioProperty should be set or get    */
} NsmDataType_e;


/**
 * The enumeration defines the different error state of Node State Manager.
 */
typedef enum _NsmErrorStatus_e
{
  NsmErrorStatus_NotSet,               /**< Initial value when error type is not set             */
  NsmErrorStatus_Ok,                   /**< Value when no error occurred                         */
  NsmErrorStatus_Error,                /**< A general, non-specific error occurred               */
  NsmErrorStatus_Dbus,                 /**< Error in D-Bus communication                         */
  NsmErrorStatus_Internal,             /**< Internal error (memory alloc. failed, etc.)          */
  NsmErrorStatus_Parameter,            /**< A passed parameter was incorrect                     */
  NsmErrorStatus_WrongSession,         /**< The requested session is unknown.                    */
  NsmErrorStatus_ResponsePending,      /**< Command accepted, return value delivered asynch.     */
  NsmErrorStatus_Last                  /**< Last error value to identify valid errors            */
} NsmErrorStatus_e;


/**
 * Possible application modes of the node.
 */
typedef enum _NsmApplicationMode_e
{
  NsmApplicationMode_NotSet,           /**< Initial state                                        */
  NsmApplicationMode_Parking,          /**< Parking          mode                                */
  NsmApplicationMode_Factory,          /**< Factory          mode                                */
  NsmApplicationMode_Transport,        /**< Transport        mode                                */
  NsmApplicationMode_Normal,           /**< Normal           mode                                */
  NsmApplicationMode_Swl,              /**< Software loading mode                                */
  NsmApplicationMode_Last              /**< Last value to identify valid values                  */
}NsmApplicationMode_e;


/**
 * The enumeration defines the different restart reasons.
 */
typedef enum _NsmRestartReason_e
{
  NsmRestartReason_NotSet,             /**< Initial value when reset reason is not set           */
  NsmRestartReason_ApplicationFailure, /**< Reset was requested by System Health Mon.            */
  NsmRestartReason_CPUOverLimit,       /**< Reset was requested by System Health Mon.            */
  NsmRestartReason_MEM_OverLimit,      /**< Reset was requested by System Health Mon.            */
  NsmRestartReason_Diagnosis,          /**< Reset was requested by diagnosis                     */
  NsmRestartReason_Swl,                /**< Reset was requested by the SWL application           */
  NsmRestartReason_User,               /**< Reset was requested by an user application           */
  NsmRestartReason_Last                /**< Last value to identify valid reset reasons           */
} NsmRestartReason_e;


/**
 * Session can be enabled seat depended.
 */
typedef enum _NsmSeat_e
{
  NsmSeat_NotSet,                      /**< Initial state                                        */
  NsmSeat_Driver,                      /**< Driver seat                                          */
  NsmSeat_CoDriver,                    /**< CoDriver seat                                        */
  NsmSeat_Rear1,                       /**< Rear 1                                               */
  NsmSeat_Rear2,                       /**< Rear 2                                               */
  NsmSeat_Rear3,                       /**< Rear 3                                               */
  NsmSeat_Last                         /**< Last valid state                                     */
}NsmSeat_e;


/**
 * The enumeration defines the different wake up reasons.
 */
typedef enum _NsmSessionState_e
{
  NsmSessionState_Unregistered,        /**< Initial state, equals "not set"                      */
  NsmSessionState_Inactive,            /**< Session is inactive                                  */
  NsmSessionState_Active               /**< Session is active                                    */
} NsmSessionState_e;

/**
 * The enumeration defines the different shutdown reasons.
 */
typedef enum _NsmShutdownReason_e
{
  NsmShutdownReason_NotSet,            /**< Initial value when ShutdownReason not set            */
  NsmShutdownReason_Normal,            /**< A normal shutdown has been performed                 */
  NsmShutdownReason_SupplyBad,         /**< Shutdown because of bad supply                       */
  NsmShutdownReason_SupplyPoor,        /**< Shutdown because of poor supply                      */
  NsmShutdownReason_ThermalBad,        /**< Shutdown because of bad thermal state                */
  NsmShutdownReason_ThermalPoor,       /**< Shutdown because of poor thermal state               */
  NsmShutdownReason_SwlNotActive,      /**< Shutdown after software loading                      */
  NsmShutdownReason_Last               /**< Last value. Identify valid ShutdownReasons           */
} NsmShutdownReason_e;

/**
 * The enumeration defines the different start or wake up reasons.
 */
typedef enum _NsmRunningReason_e
{
  NsmRunningReason_NotSet,                       /**< Initial value when reason is not set.                          */
  NsmRunningReason_WakeupCan,                    /**< Wake up because of CAN activity                                */
  NsmRunningReason_WakeupMediaEject,             /**< Wake up because of 'Eject' button                              */
  NsmRunningReason_WakeupMediaInsertion,         /**< Wake up because of media insertion                             */
  NsmRunningReason_WakeupHevac,                  /**< Wake up because of user uses the HEVAC unit in the car.
                                                      Even if the HEVAC actually causes activity on the CAN bus a
                                                      different wakeup reason is required as it could result in a
                                                      different level of functionality being started                 */
  NsmRunningReason_WakeupPhone,                  /**< Wake up because of a phone call being received.
                                                      Even if this is passed as a CAN event a different wakeup reason
                                                      is required as it could result in a different level of
                                                      functionality being started                                    */
  NsmRunningReason_WakeupPowerOnButton,          /**< Startup because user presses the "Power ON" button in the car.
                                                      Even if this is passed as a CAN event a different wakeup reason
                                                      is required as it could result in a different level of
                                                      functionality being started                                    */
  NsmRunningReason_StartupFstp,                  /**< System was started due to a first switch to power              */
  NsmRunningReason_StartupSwitchToPower,         /**< System was switched to power                                   */
  NsmRunningReason_RestartSwRequest,             /**< System was restarted due to an internal SW Request
                                                      (i.e. SWL or Diagnosis)                                        */
  NsmRunningReason_RestartInternalHealth,        /**< System was restarted due to an internal health problem         */
  NsmRunningReason_RestartExternalHealth,        /**< System was restarted due to an external health problem
                                                      (i.e. external wdog believed node was in failure)              */
  NsmRunningReason_RestartUnexpected,            /**< System was restarted due to an unexpected kernel restart.
                                                      This will be the default catch when no other reason is known   */
  NsmRunningReason_RestartUser,                  /**< Target was reset due to user action (i.e user 3 finger press)  */
  NsmRunningReason_PlatformEnd = 0x7F,           /**< Last value (127) to identify where the platform defines end
                                                      (product will start from here on)                              */
  NsmRunningReason_ProductOffset = NsmRunningReason_PlatformEnd  + 1 /**< product will start from here with index 0  */
} NsmRunningReason_e;


/**
 * The enumeration defines the different node states
 */
typedef enum _NsmNodeState_e
{
  NsmNodeState_NotSet,                 /**< Initial state when node state is not set             */
  NsmNodeState_StartUp,                /**< Basic system is starting up                          */
  NsmNodeState_BaseRunning,            /**< Basic system components have been started            */
  NsmNodeState_LucRunning,             /**< All 'Last user context' components have been started */
  NsmNodeState_FullyRunning,           /**< All 'foreground' components have been started        */
  NsmNodeState_FullyOperational,       /**< All components have been started                     */
  NsmNodeState_ShuttingDown,           /**< The system is shutting down                          */
  NsmNodeState_ShutdownDelay,          /**< Shutdown request active. System will shutdown soon   */
  NsmNodeState_FastShutdown,           /**< Fast shutdown active                                 */
  NsmNodeState_DegradedPower,          /**< Node is in degraded power state                      */
  NsmNodeState_Shutdown,               /**< Node is completely shut down                         */
  NsmNodeState_Last                    /**< Last valid entry to identify valid node states       */
} NsmNodeState_e;


/** The type defines the structure for a session.                                                */
typedef struct _NsmSession_s
{
  char               sName[NSM_MAX_SESSION_NAME_LENGTH];   /**< Name  of the session             */
  char               sOwner[NSM_MAX_SESSION_OWNER_LENGTH]; /**< Owner of the session             */
  NsmSeat_e          enSeat;                               /**< Seat  of the session             */
  NsmSessionState_e  enState;                              /**< State of the session             */
} NsmSession_s, *pNsmSession_s;


/**
 * The enumeration defines the different dimming level
 */
typedef enum _NsmDimmingLevel_e
{
  NsmDimmingLevel_Off = 0,
  NsmDimmingLevel_Degrade = 30,
  NsmDimmingLevel_On = 100
} NsmDimmingLevel_e;

/**
 * The enumeration defines the different Dimming status type.
 */
typedef enum _NsmDimmingStatus_e
{
  NsmDimmingBl_Off     = 1,                              /** Dimming back light off **/
  NsmDimmingBl_On      = 2,                              /** Dimming back light on  **/
  NsmDimmingBl_Lcm_Off = 3,                              /** Dimming back light on for LCM  **/
  NsmDimmingBl_Lcm_On  = 4                               /** Dimming back light off for LCM **/
} NsmDimmingStatus_e;

/**
 * The enumeration defines the different audio property
 */
typedef enum _NsmAudioProperty_e
{
  NsmAudioProperty_Mute = 0,
  NsmAudioProperty_Degrade = 10,
  NsmAudioProperty_Unmute = 30
} NsmAudioProperty_e;


/**
 * The enumeration defines the errors of the audiomanager.
 * All possible errors are in here.
 * This enum is used widely as return parameter.
 */
enum NSM_AudioError_e
{

    NSM_AM_E_OK = 0,                                         /** Audio: no error - positive reply **/
    NSM_AM_E_UNKNOWN = 1,                                    /** Audio: default                   **/
    NSM_AM_E_OUT_OF_RANGE = 2,                               /** Audio: value out of range        **/
    NSM_AM_E_NOT_USED = 3,                                   /** Audio: not used                  **/
    NSM_AM_E_DATABASE_ERROR = 4,                             /** Audio: a database error occurred **/
    NSM_AM_E_ALREADY_EXISTS = 5,                             /** Audio: the desired object already exists   **/
    NSM_AM_E_NO_CHANGE = 6,                                  /** Audio: there is no change                  **/
    NSM_AM_E_NOT_POSSIBLE = 7,                               /** Audio: the desired action is not possible  **/
    NSM_AM_E_NON_EXISTENT = 8,                               /** Audio: the desired object is non existent  **/
    NSM_AM_E_ABORTED = 9,                                    /** Audio: the asynchronous action was aborted **/

    NSM_AM_E_WRONG_FORMAT = 10,                              /**
                                                              * This error is returned in case a connect is issued with a connectionFormat that
                                                              * cannot be selected for the connection. This could be either due to the
                                                              * capabilities of a source or a sink or gateway compatibilities for example
                                                              */
    NSM_AM_E_MAX
};

/**
 * The enumeration defines the different reset reason
 */
typedef enum _NsmResetReason_e
{
  NsmResetReason_NotSet                    = 0,
  NsmResetReason_Shutdown_ACC              = NSM_BIT_VALUE(0),
  NsmResetReason_Shutdown_BadVoltage       = NSM_BIT_VALUE(1),
  NsmResetReason_Shutdown_BadTemp          = NSM_BIT_VALUE(2),

  /* Add other reason here... */
  NsmResetReason_Unknown                   = NSM_BIT_VALUE(30),
  NsmResetReason_Invalid                   = NSM_BIT_VALUE(31)
} NsmResetReason_e;

/**
 * The enumeration defines the different persistent data type
 */
typedef enum _NsmPSDataType_e
{
  NsmPSDataType_INT8,
  NsmPSDataType_UINT8,
  NsmPSDataType_INT16,
  NsmPSDataType_UINT16,
  NsmPSDataType_INT32,
  NsmPSDataType_UINT32,
  NsmPSDataType_BOOLEAN,
  NsmPSDataType_STRING,
  NsmPSDataType_STREAM,
  NsmPSDataType_INVALID

} NsmPSDataType_e;

/* The enumeration defines the different error state of Critical Voltage Manager.*/
enum CvmErrorStatus_e
{
  CVM_ERROR_STATUS_NOT_SET            = 0,    /* Initial value when error type is not set */
  CVM_ERROR_STATUS_OK,                        /* Value when no error occurred             */
  CVM_ERROR_STATUS_ERROR,                     /* A general, non-specific error occurred   */
  CVM_ERROR_STATUS_INVALID_PARAM,             /* The invalid parameter(s) detected        */
  CVM_ERROR_STATUS_MAX
};


/*
    The enumeration defines the different state of the battery that can be exchanged between the NodeStateManager (NSM)
    and the CriticalVoltageManager (CVM). Based on this value, the setter and getter functions of the NSM and CVM will
    interpret data behind the passed byte pointer.
*/
enum CvmBatteryStatus_e
{
  CVM_BATTERY_STATUS_NOT_SET          = 0x00, /* Initial state when battery state is not set              */
  CVM_BATTERY_STATUS_GOOD             = 0x01, /* The battery's voltage is between 9V and 16V              */
  CVM_BATTERY_STATUS_POOR             = 0x02, /* The battery's voltage is between 6V ~ 9V and 16V ~ 17.2V */
  CVM_BATTERY_STATUS_BAD              = 0x03, /* The battery's voltage is below 6V and above 17.2V        */
  CVM_BATTERY_STATUS_DRIVER_TIMEOUT   = 0x04, /* The battery's driver failed                              */
  CVM_BATTERY_STATUS_MAX
};


/**
* The enumeration defines the different thermal Main State type.
*/
enum HcmThermalMainState_e
{
  /* Below lists main states definition
  ** Only for other modules use
  */
  HCM_THERMAL_GOOD    =   0,          /**< Thermal Good State- Mainstate 1 >**/
  HCM_THERMAL_POOR,                   /**< Thermal Poor State- Mainstate 2 >**/
  HCM_THERMAL_BAD,                    /**< Thermal Bad  State- Mainstate 3 >**/
};

/*
    The enumeration defines the different state of the system memory that can be exchanged between NodeStateManager (NSM)
    and the NodeHealthMonitor (NHM). Based on this value, the setter and getter functions of the NSM and NHM will
    interpret data behind the passed byte pointer.
*/
enum NhmSysMemState_e
{
  NHM_SYS_MEM_STATE_NORMAL    = 0,    /* The usage of the system memory is below 80%       */
  NHM_SYS_MEM_STATE_HIGH,             /* The usage of the system memory is above 80%       */
  NHM_SYS_MEM_STATE_NOT_SET           /* Initial state when system memory state is not set */
};

/*
    The enumeration defines the different state of the system memory that can be exchanged between NodeStateManager (NSM)
    and the NodeHealthMonitor (NHM). Based on this value, the setter and getter functions of the NSM and NHM will
    interpret data behind the passed byte pointer.
*/
enum NhmOverMemoryRequest_e
{
  NHM_OM_REQ_NONE             = 0,    /* No Action when the system memory is in high usage */
  NHM_OM_REQ_REBOOT,                  /* Reboot    when the system memory is in high usage */
  NHM_OM_REQ_NOT_SET
};

/* This enum will be used to report the status of an application before/after or during a failure */
//enum NhmAppStatus_e
//{
//  NHM_APP_STATUS_FAILED,     /**< Used when an application has failed                                      */
//  NHM_APP_STATUS_RESTARTING, /**< Used when an application has failed but is in process of being restarted */
//  NHM_APP_STATUS_OK          /**< Used when an application failed but has correctly been restarted         */
//};
//
//
///* This enum will be used for indicating the status of method calls */
//enum NhmErrorStatus_e
//{
//  NHM_ERROR_STATUS_OK,                  /**< This value will be used to state that the method worked as expected                                  */
//  NHM_ERROR_STATUS_ERROR,               /**< This value can be used to state that an error occurred handling the request                          */
//  NHM_ERROR_STATUS_UNKNOWN_APP,         /**< This value will be set when the passed string does not correspond to a failed application            */
//  NHM_ERROR_STATUS_RESTART_NOT_POSSIBLE /**< This value will be used when an application requests a node restart but it is not currently possible */
//};


#ifdef __cplusplus
}
#endif
/** \} */ /* End of SSW_NSM_INTERFACE */
/** \} */ /* End of SSW_NSM_TEMPLATE  */
#endif /* NODESTATETYPES_H */
